/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.common.config;

import io.confluent.common.config.ConfigException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ConfigDef {
    private static final Object NO_DEFAULT_VALUE = new String("");
    private final Map<String, ConfigKey> configKeys = new HashMap<String, ConfigKey>();

    public Set<String> names() {
        return Collections.unmodifiableSet(this.configKeys.keySet());
    }

    public ConfigDef define(String name, Type type, Object defaultValue, Validator validator, Importance importance, String documentation) {
        if (this.configKeys.containsKey(name)) {
            throw new ConfigException("Configuration " + name + " is defined twice.");
        }
        Object parsedDefault = defaultValue == NO_DEFAULT_VALUE ? NO_DEFAULT_VALUE : this.parseType(name, defaultValue, type);
        this.configKeys.put(name, new ConfigKey(name, type, parsedDefault, validator, importance, documentation));
        return this;
    }

    public ConfigDef defineOverride(String name, Type type, Object defaultValue, Validator validator, Importance importance, String documentation) {
        if (!this.configKeys.containsKey(name)) {
            throw new ConfigException("Configuration " + name + " is defined as an override but does not exist.");
        }
        Object parsedDefault = defaultValue == NO_DEFAULT_VALUE ? NO_DEFAULT_VALUE : this.parseType(name, defaultValue, type);
        this.configKeys.put(name, new ConfigKey(name, type, parsedDefault, validator, importance, documentation));
        return this;
    }

    public ConfigDef define(String name, Type type, Object defaultValue, Importance importance, String documentation) {
        return this.define(name, type, defaultValue, null, importance, documentation);
    }

    public ConfigDef defineOverride(String name, Type type, Object defaultValue, Importance importance, String documentation) {
        return this.defineOverride(name, type, defaultValue, null, importance, documentation);
    }

    public ConfigDef define(String name, Type type, Validator validator, Importance importance, String documentation) {
        return this.define(name, type, NO_DEFAULT_VALUE, validator, importance, documentation);
    }

    public ConfigDef defineOverride(String name, Type type, Validator validator, Importance importance, String documentation) {
        return this.defineOverride(name, type, NO_DEFAULT_VALUE, validator, importance, documentation);
    }

    public ConfigDef define(String name, Type type, Importance importance, String documentation) {
        return this.define(name, type, NO_DEFAULT_VALUE, null, importance, documentation);
    }

    public ConfigDef defineOverride(String name, Type type, Importance importance, String documentation) {
        return this.defineOverride(name, type, NO_DEFAULT_VALUE, null, importance, documentation);
    }

    public Map<String, Object> parse(Map<?, ?> props) {
        HashMap<String, Object> values = new HashMap<String, Object>();
        for (ConfigKey key : this.configKeys.values()) {
            Object value;
            if (props.containsKey(key.name)) {
                value = this.parseType(key.name, props.get(key.name), key.type);
            } else {
                if (key.defaultValue == NO_DEFAULT_VALUE) {
                    throw new ConfigException("Missing required configuration \"" + key.name + "\" which has no default value.");
                }
                value = key.defaultValue;
            }
            if (key.validator != null) {
                key.validator.ensureValid(key.name, value);
            }
            values.put(key.name, value);
        }
        return values;
    }

    private Object parseType(String name, Object value, Type type) {
        try {
            String trimmed = null;
            if (value instanceof String) {
                trimmed = ((String)value).trim();
            }
            switch (type) {
                case BOOLEAN: {
                    if (value instanceof String) {
                        if (trimmed.equalsIgnoreCase("true")) {
                            return true;
                        }
                        if (trimmed.equalsIgnoreCase("false")) {
                            return false;
                        }
                        throw new ConfigException(name, value, "Expected value to be either true or false");
                    }
                    if (value instanceof Boolean) {
                        return value;
                    }
                    throw new ConfigException(name, value, "Expected value to be either true or false");
                }
                case STRING: {
                    if (value instanceof String) {
                        return trimmed;
                    }
                    throw new ConfigException(name, value, "Expected value to be a string, but it was a " + value.getClass().getName());
                }
                case INT: {
                    if (value instanceof Integer) {
                        return (Integer)value;
                    }
                    if (value instanceof String) {
                        return Integer.parseInt(trimmed);
                    }
                    throw new ConfigException(name, value, "Expected value to be an number.");
                }
                case LONG: {
                    if (value instanceof Integer) {
                        return ((Integer)value).longValue();
                    }
                    if (value instanceof Long) {
                        return (Long)value;
                    }
                    if (value instanceof String) {
                        return Long.parseLong(trimmed);
                    }
                    throw new ConfigException(name, value, "Expected value to be an number.");
                }
                case DOUBLE: {
                    if (value instanceof Number) {
                        return ((Number)value).doubleValue();
                    }
                    if (value instanceof String) {
                        return Double.parseDouble(trimmed);
                    }
                    throw new ConfigException(name, value, "Expected value to be an number.");
                }
                case LIST: {
                    if (value instanceof List) {
                        return (List)value;
                    }
                    if (value instanceof String) {
                        if (trimmed.isEmpty()) {
                            return Collections.emptyList();
                        }
                        return Arrays.asList(trimmed.split("\\s*,\\s*", -1));
                    }
                    throw new ConfigException(name, value, "Expected a comma separated list.");
                }
                case CLASS: {
                    if (value instanceof Class) {
                        return (Class)value;
                    }
                    if (value instanceof String) {
                        return Class.forName(trimmed);
                    }
                    throw new ConfigException(name, value, "Expected a Class instance or class name.");
                }
            }
            throw new IllegalStateException("Unknown type.");
        }
        catch (NumberFormatException e) {
            throw new ConfigException(name, value, "Not a number of type " + (Object)((Object)type));
        }
        catch (ClassNotFoundException e) {
            throw new ConfigException(name, value, "Class " + value + " could not be found.");
        }
    }

    public String toHtmlTable() {
        List<ConfigKey> configs = this.sortedConfigs();
        StringBuilder b = new StringBuilder();
        b.append("<table>\n");
        b.append("<tr>\n");
        b.append("<th>Name</th>\n");
        b.append("<th>Type</th>\n");
        b.append("<th>Default</th>\n");
        b.append("<th>Importance</th>\n");
        b.append("<th>Description</th>\n");
        b.append("</tr>\n");
        for (ConfigKey def : configs) {
            b.append("<tr>\n");
            b.append("<td>");
            b.append(def.name);
            b.append("</td>");
            b.append("<td>");
            b.append(def.type.toString().toLowerCase());
            b.append("</td>");
            b.append("<td>");
            b.append(def.defaultValue == null ? "" : def.defaultValue);
            b.append("</td>");
            b.append("<td>");
            b.append(def.importance.toString().toLowerCase());
            b.append("</td>");
            b.append("<td>");
            b.append(def.documentation);
            b.append("</td>");
            b.append("</tr>\n");
        }
        b.append("</table>");
        return b.toString();
    }

    public String toRst() {
        List<ConfigKey> configs = this.sortedConfigs();
        StringBuilder b = new StringBuilder();
        for (ConfigKey def : configs) {
            b.append("``");
            b.append(def.name);
            b.append("``\n");
            for (String docLine : def.documentation.split("\n")) {
                if (docLine.length() == 0) continue;
                b.append("  ");
                b.append(docLine);
                b.append("\n\n");
            }
            b.append("  * Type: ");
            b.append(def.type.toString().toLowerCase());
            b.append("\n");
            if (def.defaultValue != null) {
                b.append("  * Default: ");
                if (def.type == Type.STRING) {
                    b.append("\"");
                    b.append(def.defaultValue);
                    b.append("\"");
                } else {
                    b.append(def.defaultValue);
                }
                b.append("\n");
            }
            b.append("  * Importance: ");
            b.append(def.importance.toString().toLowerCase());
            b.append("\n\n");
        }
        return b.toString();
    }

    private List<ConfigKey> sortedConfigs() {
        ArrayList<ConfigKey> configs = new ArrayList<ConfigKey>(this.configKeys.values());
        Collections.sort(configs, new Comparator<ConfigKey>(){

            @Override
            public int compare(ConfigKey k1, ConfigKey k2) {
                if (!k1.hasDefault() && k2.hasDefault()) {
                    return -1;
                }
                if (!k2.hasDefault() && k1.hasDefault()) {
                    return 1;
                }
                int cmp = k1.importance.compareTo(k2.importance);
                if (cmp == 0) {
                    return k1.name.compareTo(k2.name);
                }
                return cmp;
            }
        });
        return configs;
    }

    private static class ConfigKey {
        public final String name;
        public final Type type;
        public final String documentation;
        public final Object defaultValue;
        public final Validator validator;
        public final Importance importance;

        public ConfigKey(String name, Type type, Object defaultValue, Validator validator, Importance importance, String documentation) {
            this.name = name;
            this.type = type;
            this.defaultValue = defaultValue;
            this.validator = validator;
            this.importance = importance;
            if (this.validator != null) {
                this.validator.ensureValid(name, defaultValue);
            }
            this.documentation = documentation;
        }

        public boolean hasDefault() {
            return this.defaultValue != NO_DEFAULT_VALUE;
        }
    }

    public static class ValidString
    implements Validator {
        List<String> validStrings;

        private ValidString(List<String> validStrings) {
            this.validStrings = validStrings;
        }

        public static ValidString in(List<String> validStrings) {
            return new ValidString(validStrings);
        }

        @Override
        public void ensureValid(String name, Object o) {
            String s = (String)o;
            if (!this.validStrings.contains(s)) {
                throw new ConfigException(name, o, "String must be one of:" + this.join(this.validStrings));
            }
        }

        public String toString() {
            return "[" + this.join(this.validStrings) + "]";
        }

        private String join(List<String> list) {
            StringBuilder sb = new StringBuilder();
            boolean first = true;
            for (String item : list) {
                if (first) {
                    first = false;
                } else {
                    sb.append(",");
                }
                sb.append(item);
            }
            return sb.toString();
        }
    }

    public static class Range
    implements Validator {
        private final Number min;
        private final Number max;

        private Range(Number min, Number max) {
            this.min = min;
            this.max = max;
        }

        public static Range atLeast(Number min) {
            return new Range(min, null);
        }

        public static Range between(Number min, Number max) {
            return new Range(min, max);
        }

        @Override
        public void ensureValid(String name, Object o) {
            Number n = (Number)o;
            if (this.min != null && n.doubleValue() < this.min.doubleValue()) {
                throw new ConfigException(name, o, "Value must be at least " + this.min);
            }
            if (this.max != null && n.doubleValue() > this.max.doubleValue()) {
                throw new ConfigException(name, o, "Value must be no more than " + this.max);
            }
        }

        public String toString() {
            if (this.min == null) {
                return "[...," + this.max + "]";
            }
            if (this.max == null) {
                return "[" + this.min + ",...]";
            }
            return "[" + this.min + ",...," + this.max + "]";
        }
    }

    public static interface Validator {
        public void ensureValid(String var1, Object var2);
    }

    public static enum Importance {
        HIGH,
        MEDIUM,
        LOW;

    }

    public static enum Type {
        BOOLEAN,
        STRING,
        INT,
        LONG,
        DOUBLE,
        LIST,
        CLASS;

    }
}

