/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.yarn;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.AbstractIdleService;
import com.typesafe.config.Config;
import java.io.File;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.cluster.GobblinHelixMessagingService;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.gobblin.yarn.HelixMessageSubTypes;
import org.apache.gobblin.yarn.YarnHelixUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.helix.Criteria;
import org.apache.helix.HelixManager;
import org.apache.helix.InstanceType;
import org.apache.helix.model.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class YarnAppSecurityManager
extends AbstractIdleService {
    private static final Logger LOGGER = LoggerFactory.getLogger(YarnAppSecurityManager.class);
    private final Config config;
    private final HelixManager helixManager;
    private final FileSystem fs;
    private final Path tokenFilePath;
    private UserGroupInformation loginUser;
    private Token<? extends TokenIdentifier> token;
    private final long loginIntervalInMinutes;
    private final long tokenRenewIntervalInMinutes;
    private final ScheduledExecutorService loginExecutor;
    private final ScheduledExecutorService tokenRenewExecutor;
    private Optional<ScheduledFuture<?>> scheduledTokenRenewTask = Optional.absent();
    private volatile boolean firstLogin = true;

    public YarnAppSecurityManager(Config config, HelixManager helixManager, FileSystem fs, Path tokenFilePath) throws IOException {
        this.config = config;
        this.helixManager = helixManager;
        this.fs = fs;
        this.tokenFilePath = tokenFilePath;
        this.fs.makeQualified(tokenFilePath);
        this.loginUser = UserGroupInformation.getLoginUser();
        this.loginIntervalInMinutes = config.getLong("gobblin.yarn.login.interval.minutes");
        this.tokenRenewIntervalInMinutes = config.getLong("gobblin.yarn.token.renew.interval.minutes");
        this.loginExecutor = Executors.newSingleThreadScheduledExecutor(ExecutorsUtils.newThreadFactory((Optional)Optional.of((Object)LOGGER), (Optional)Optional.of((Object)"KeytabReLoginExecutor")));
        this.tokenRenewExecutor = Executors.newSingleThreadScheduledExecutor(ExecutorsUtils.newThreadFactory((Optional)Optional.of((Object)LOGGER), (Optional)Optional.of((Object)"TokenRenewExecutor")));
    }

    protected void startUp() throws Exception {
        LOGGER.info("Starting the " + YarnAppSecurityManager.class.getSimpleName());
        LOGGER.info(String.format("Scheduling the login task with an interval of %d minute(s)", this.loginIntervalInMinutes));
        this.loginExecutor.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                try {
                    if (YarnAppSecurityManager.this.scheduledTokenRenewTask.isPresent() && ((ScheduledFuture)YarnAppSecurityManager.this.scheduledTokenRenewTask.get()).cancel(true)) {
                        LOGGER.info("Cancelled the token renew task");
                    }
                    YarnAppSecurityManager.this.loginFromKeytab();
                    if (YarnAppSecurityManager.this.firstLogin) {
                        YarnAppSecurityManager.this.firstLogin = false;
                    }
                    YarnAppSecurityManager.this.scheduleTokenRenewTask();
                }
                catch (IOException ioe) {
                    LOGGER.error("Failed to login from keytab", (Throwable)ioe);
                    throw Throwables.propagate((Throwable)ioe);
                }
            }
        }, 0L, this.loginIntervalInMinutes, TimeUnit.MINUTES);
    }

    protected void shutDown() throws Exception {
        LOGGER.info("Stopping the " + YarnAppSecurityManager.class.getSimpleName());
        if (this.scheduledTokenRenewTask.isPresent()) {
            ((ScheduledFuture)this.scheduledTokenRenewTask.get()).cancel(true);
        }
        ExecutorsUtils.shutdownExecutorService((ExecutorService)this.loginExecutor, (Optional)Optional.of((Object)LOGGER));
        ExecutorsUtils.shutdownExecutorService((ExecutorService)this.tokenRenewExecutor, (Optional)Optional.of((Object)LOGGER));
    }

    private void scheduleTokenRenewTask() {
        LOGGER.info(String.format("Scheduling the token renew task with an interval of %d minute(s)", this.tokenRenewIntervalInMinutes));
        this.scheduledTokenRenewTask = Optional.of(this.tokenRenewExecutor.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                try {
                    YarnAppSecurityManager.this.renewDelegationToken();
                }
                catch (IOException ioe) {
                    LOGGER.error("Failed to renew delegation token", (Throwable)ioe);
                    throw Throwables.propagate((Throwable)ioe);
                }
                catch (InterruptedException ie) {
                    LOGGER.error("Token renew task has been interrupted");
                    Thread.currentThread().interrupt();
                }
            }
        }, this.tokenRenewIntervalInMinutes, this.tokenRenewIntervalInMinutes, TimeUnit.MINUTES));
    }

    private synchronized void renewDelegationToken() throws IOException, InterruptedException {
        this.token.renew(this.fs.getConf());
        this.writeDelegationTokenToFile();
        if (!this.firstLogin) {
            this.sendTokenFileUpdatedMessage(InstanceType.CONTROLLER);
            this.sendTokenFileUpdatedMessage(InstanceType.PARTICIPANT);
        }
    }

    @VisibleForTesting
    synchronized void getNewDelegationTokenForLoginUser() throws IOException {
        this.token = this.fs.getDelegationToken(this.loginUser.getShortUserName());
    }

    private void loginFromKeytab() throws IOException {
        String keyTabFilePath = this.config.getString("gobblin.yarn.keytab.file.path");
        if (Strings.isNullOrEmpty((String)keyTabFilePath)) {
            throw new IOException("Keytab file path is not defined for Kerberos login");
        }
        if (!new File(keyTabFilePath).exists()) {
            throw new IOException("Keytab file not found at: " + keyTabFilePath);
        }
        String principal = this.config.getString("gobblin.yarn.keytab.principal.name");
        if (Strings.isNullOrEmpty((String)principal)) {
            principal = this.loginUser.getShortUserName() + "/localhost@LOCALHOST";
        }
        Configuration conf = new Configuration();
        conf.set("hadoop.security.authentication", UserGroupInformation.AuthenticationMethod.KERBEROS.toString().toLowerCase());
        UserGroupInformation.setConfiguration((Configuration)conf);
        UserGroupInformation.loginUserFromKeytab((String)principal, (String)keyTabFilePath);
        LOGGER.info(String.format("Logged in from keytab file %s using principal %s", keyTabFilePath, principal));
        this.loginUser = UserGroupInformation.getLoginUser();
        this.getNewDelegationTokenForLoginUser();
        this.writeDelegationTokenToFile();
        if (!this.firstLogin) {
            this.sendTokenFileUpdatedMessage(InstanceType.CONTROLLER);
            this.sendTokenFileUpdatedMessage(InstanceType.PARTICIPANT);
        }
    }

    @VisibleForTesting
    synchronized void writeDelegationTokenToFile() throws IOException {
        if (this.fs.exists(this.tokenFilePath)) {
            LOGGER.info("Deleting existing token file " + this.tokenFilePath);
            this.fs.delete(this.tokenFilePath, false);
        }
        LOGGER.info("Writing new or renewed token to token file " + this.tokenFilePath);
        YarnHelixUtils.writeTokenToFile(this.token, this.tokenFilePath, this.fs.getConf());
        this.fs.setPermission(this.tokenFilePath, new FsPermission(FsAction.READ_WRITE, FsAction.NONE, FsAction.NONE));
    }

    @VisibleForTesting
    void sendTokenFileUpdatedMessage(InstanceType instanceType) {
        Criteria criteria = new Criteria();
        criteria.setInstanceName("%");
        criteria.setResource("%");
        criteria.setPartition("%");
        criteria.setPartitionState("%");
        criteria.setRecipientInstanceType(instanceType);
        criteria.setSessionSpecific(true);
        Message tokenFileUpdatedMessage = new Message(Message.MessageType.USER_DEFINE_MSG, HelixMessageSubTypes.TOKEN_FILE_UPDATED.toString().toLowerCase() + UUID.randomUUID().toString());
        tokenFileUpdatedMessage.setMsgSubType(HelixMessageSubTypes.TOKEN_FILE_UPDATED.toString());
        tokenFileUpdatedMessage.setMsgState(Message.MessageState.NEW);
        if (instanceType == InstanceType.CONTROLLER) {
            tokenFileUpdatedMessage.setTgtSessionId("*");
        }
        GobblinHelixMessagingService messagingService = new GobblinHelixMessagingService(this.helixManager);
        int messagesSent = messagingService.send(criteria, tokenFileUpdatedMessage);
        LOGGER.info(String.format("Sent %d token file updated message(s) to the %s", messagesSent, instanceType));
    }
}

