/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.skywalking.apm.plugin.zookeeper;

import org.apache.skywalking.apm.agent.core.context.ContextManager;
import org.apache.skywalking.apm.agent.core.context.tag.Tags;
import org.apache.skywalking.apm.agent.core.context.trace.AbstractSpan;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.EnhancedInstance;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.InstanceMethodsAroundInterceptor;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.MethodInterceptResult;
import org.apache.skywalking.apm.network.trace.component.ComponentsDefine;
import org.apache.zookeeper.WatchedEvent;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

public class EventThreadMethodInterceptor implements InstanceMethodsAroundInterceptor {

    @Override
    public void beforeMethod(EnhancedInstance objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes,
        MethodInterceptResult result) throws Throwable {
        if (isWatchedEvent(allArguments[0])) {
            Field field = allArguments[0].getClass().getDeclaredField("event");
            field.setAccessible(true);
            WatchedEvent event = (WatchedEvent) field.get(allArguments[0]);
            AbstractSpan span = ContextManager.createEntrySpan("Zookeeper/WatchedEvent/" + event.getType()
                                                                                                .name(), null);
            ZooOpt.setTags(span, event);
            span.setComponent(ComponentsDefine.ZOOKEEPER);
            Tags.DB_TYPE.set(span, "Zookeeper");
        }
    }

    @Override
    public Object afterMethod(EnhancedInstance objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes,
        Object ret) throws Throwable {
        if (isWatchedEvent(allArguments[0])) {
            ContextManager.stopSpan();
        }
        return ret;
    }

    @Override
    public void handleMethodException(EnhancedInstance objInst, Method method, Object[] allArguments,
        Class<?>[] argumentsTypes, Throwable t) {
        if (isWatchedEvent(allArguments[0])) {
            ContextManager.activeSpan().log(t);
        }
    }

    private boolean isWatchedEvent(Object event) {
        return event != null && "org.apache.zookeeper.ClientCnxn$WatcherSetEventPair".equals(event.getClass()
                                                                                                  .getName());
    }
}
